;*
;* JAVA MEMORY MANAGER FOR 6502
;*
;*
;* HANDLE TABLE IS A TABLE OF POINTERS TO MEMORY BLOCKS
;*
;* THE LOW ORDER 3 BITS ARE STATE BITS FOR THE MEMORY BLOCK
;*    1 (001) = IN USE
;*    3 (011) = FIXED   - HAS TO REMAIN IN PLACE
;*    5 (101) = CODE
;*    7 (111) = SWAPPED - NEEDS TO BE RELOADED FROM SOURCE OR SWAP
;*
;* 
;* MEMORY BLOCKS ARE ALIGNED ON 64 BIT BOUNDARIES
;*
;*
;* A MEMORY HANDLE IS A POINTER TO THE ENTRY IN THE TABLE
;*
;* AN ALLOCATED MEMORY BLOCK IS ORGANIZED AS:
;*    OFFSET  SIZE  FUNCTION
;*       0      2     SIZE
;*       2      2    REF CNT - MSB = ACCESSED BIT
;*       4      ?     DATA
;* A FREE MEMORY BLOCK IS ORGANIZED AS:
;*    OFFSET  SIZE  FUNCTION
;*       0      2     SIZE
;*       2      2     HNEXT 
	.INCLUDE	"global.inc"
	.IMPORT	INIT_START,INIT_END,WARM_INIT
	.IMPORT	PRNTAX,COUT,CROUT,PUTS,PUTSLN,PRBYTE,PRSTR,PRSTRLN,MEMCPY,MEMCLR,KBWAIT
	.IMPORT	SYSTHROW,THROW_INTERNALERR
	.EXPORT	HMEM_INIT
	.EXPORT	HMEM_ALLOC,HMEM_ALLOC_CODE,HMEM_ALLOC_FIXED,HMEM_FREE
	.EXPORT	HMEM_PTR,HMEM_REF_INC,HMEM_REF_DEC,HMEM_CLR
	.EXPORT	HMEM_LOCK,HMEM_UNLOCK,HMEM_UNLOCK_CODE,HMEM_GC
	.EXPORT	HMEM_COALESCE,HMEM_COMPACT

	.SEGMENT "INIT"
;*
;* CREATE EXTERNAL LINKEAGE TABLE AT $302
;*
HMEM_INIT:	LDA	#<HMEM_PTR
	STA	LINK_HMEMPTR
	LDA	#>HMEM_PTR
	STA	LINK_HMEMPTR+1
	LDA	#<HMEM_ALLOC
	STA	LINK_HMEMALLOC
	LDA	#>HMEM_ALLOC
	STA	LINK_HMEMALLOC+1
	LDA	#<HMEM_ALLOC_FIXED
	STA	LINK_HMEMALLOCFIXED
	LDA	#>HMEM_ALLOC_FIXED
	STA	LINK_HMEMALLOCFIXED+1
	LDA	#<HMEM_FREE
	STA	LINK_HMEMFREE
	LDA	#>HMEM_FREE
	STA	LINK_HMEMFREE+1
	LDA	#<HMEM_LOCK
	STA	LINK_HMEMLOCK
	LDA	#>HMEM_LOCK
	STA	LINK_HMEMLOCK+1
	LDA	#<HMEM_UNLOCK
	STA	LINK_HMEMUNLOCK
	LDA	#>HMEM_UNLOCK
	STA	LINK_HMEMUNLOCK+1
	LDA	#<HMEM_REF_INC
	STA	LINK_HMEMREFINC
	LDA	#>HMEM_REF_INC
	STA	LINK_HMEMREFINC+1
	LDA	#<HMEM_REF_DEC
	STA	LINK_HMEMREFDEC
	LDA	#>HMEM_REF_DEC
	STA	LINK_HMEMREFDEC+1
	LDA	#<HMEM_GC
	STA	LINK_GC
	LDA	#>HMEM_GC
	STA	LINK_GC+1
;*
;* INITIALIZE THE HANDLE TABLE
;*
	LDY	#$00
	STY	GCNEEDED
	LDX	#<HTBL
	STX	HUNUSED
	LDA	#>HTBL
	STA	HUNUSED+1
LINK_UNUSED:	STX	HNDL
	STA	HNDL+1
	LDA	HNDL
	CLC			; MAKE UNUSED HANDLE LIST
	ADC	#$02
	STA	(HNDL),Y
	TAX
	INY
	LDA	HNDL+1
	ADC	#$00
	STA	(HNDL),Y
	DEY
	CMP	#>HTBL_END
	BNE	LINK_UNUSED
	TYA
	STA	(HNDL),Y		; NULL OUT LAST IN LIST
	INY
	STA	(HNDL),Y
	JSR	HMEM_NEW		; GET NEXT AVAIL HANDLE
	STA	HFREE		; MAKE INITAL FREE BLOCK
	STX	HFREE+1
.IFDEF	DEBUG_MEMMGR
	STA	HFREEBAK		; MAKE INITAL FREE BLOCK
	STX	HFREEBAK+1
.ENDIF	
	LDY	#$00		; SET FREE MEM PTR
	LDA	#<INIT_START
	CLC
	ADC	#$07
	AND	#$F8
	STA	MPTR
	STA	(HFREE),Y
	INY
	LDA	#>INIT_START
	ADC	#$00
	STA	MPTR+1
	STA	(HFREE),Y
	DEY			; SET FREE MEM SIZE
	LDA	#<HTBL
	SEC
	SBC	MPTR
	STA	(MPTR),Y
	INY
	LDA	#>HTBL
	SBC	MPTR+1
	STA	(MPTR),Y
	INY
	LDA	#$00		; NULL OUT NEXT FREE HANDLE
	STA	(MPTR),Y
	INY
	STA	(MPTR),Y
	LDA	#'$'
	JSR	COUT
	LDY	#$00
	LDA	(MPTR),Y
	INY
	TAX
	LDA	(MPTR),Y
	JSR	PRNTAX
	PSTRLN	" bytes available"
.IFDEF	DEBUG_MEMMGR
	JSR	HMEM_DUMP
	JSR	CROUT
.ENDIF
.IFDEF	SWAPPING
MAKESWAP:	LDX	CFG_SWAPVOL
	BNE	HAVESWAP
	JSR	PRODOS		; FIND ONLINE VOLS
	.BYTE	$C5
	.ADDR	ONLINE_PARAMS
	BCS	HAVESWAP		; USE DEFAULT VOLUME ON ERR
	INX			; PREPEND '/'
	STX	CFG_SWAPVOL
	LDX	#'/'
	STX	CFG_SWAPVOL+1
	STA	TMP		; SAVE VOL COUNT
	LDX	#$00
GETVOLS:	LDA	VOLNAMES, X
	INX
	AND	#$0F
	BEQ	NEXTVOL
	STA	TMP+1		; COPY VOL NAME
	LDY	#$00
:	LDA	VOLNAMES, X
	INX
	STA	CFG_SWAPVOL+2,Y
	INY
	CPY	TMP+1
	BNE	:-
	INY
	STY	CFG_SWAPVOL		; SAVE VOL NAME LENGTH
	DEX
NEXTVOL:	TXA			; MOVE TO NEXT ONLINE VOL	
	CLC
	ADC	#$0F
	AND	#$F0
	TAX
	DEC	TMP
	BNE	GETVOLS
HAVESWAP:	LDX	CFG_SWAPVOL		; COPY VOL NAME
COPYVOL:	LDA	CFG_SWAPVOL,X
	STA	SWAPFILE,X
	DEX
	BPL	COPYVOL
	TAX			; COPY SWAPFILE NAME
	LDY	#$00
COPYNAME:	LDA	SWAPNAME+1,Y
	STA	SWAPFILE+1,X
	INX
	INY
	CPY	SWAPNAME
	BNE	COPYNAME
	STX	SWAPFILE
	JSR	PRODOS		; DESTROY EXISTING SWAPFILE
	.BYTE	$C1
	.ADDR	DESTROY_PARAMS
	JSR	PRODOS
	.BYTE	$C0		; CREATE SWAPFILE
	.ADDR	CREATE_PARAMS
	RTS
;*
;* INIT-TIME DATA
;*
CFG_SWAPVOL:	.BYTE	0,0		; TEMPORARY!!!

VOLNAMES:	.RES	256
SWAPNAME:	.BYTE	9,"/SWAPFILE"
ONLINE_PARAMS:	.BYTE	2
	.BYTE	0
	.ADDR	VOLNAMES
DESTROY_PARAMS:	.BYTE	1
	.ADDR	SWAPFILE
CREATE_PARAMS:	.BYTE	7
	.ADDR	SWAPFILE		; PATHNAME
	.BYTE	$83		; ACCESS BITS
	.BYTE	$06		; FILE TYPE = BIN
	.WORD	$0000		; AUX TYPE = NOP
	.BYTE	1		; STORAGE TYPE
	.WORD	0		; DATE
	.WORD	0		; TIME

	.DATA
SWAPFILE:	.RES	24
SWAPLEN:	.BYTE	0,0,0

.ELSE
	RTS
.ENDIF
;*
;* MEMORY MANAGER
;*

	.CODE
;*
;* CONVERT HANDLE TO POINTER
;* ENTRYY: AX = HANDLE
;* EXIT:   AX = POINTER
;*
HMEM_PTR:	STA	HNDL
	STX	HNDL+1
.IFDEF	DEBUG
	JSR	HMEM_VALIDATE
.ENDIF
	LDY	#$01
	LDA	(HNDL),Y
	DEY
	TAX
	LDA	(HNDL),Y
.IFDEF	SWAPPING
	TAY
	AND	#$07
	CMP	#$07
	BEQ	HMEM_PTR_SWAPIN
	TYA
	AND	#$F8
	STA	MPTR
	STX	MPTR+1
	LDY	#$03
	LDA	(MPTR),Y
	ORA	#$80		; SET ACCESSED FLAG
	STA	(MPTR),Y
	LDA	MPTR
.ELSE
	AND	#$F8
.ENDIF
	ORA	#$04		; RETURN POINTER, NO ERR
	CLC
	RTS
.IFDEF	SWAPPING
HMEM_PTR_SWAPIN: TXA
	STA	MPTR
	INY
	LDA	(HNDL),Y
	STA	MPTR+1
	LDY	#$04
	LDA	(MPTR),Y		; GET POINTER TO SWAPIN ROUTINE
	STA	OPJMP
	INY
	LDA	(MPTR),Y
	STA	OPJMP+1
	LDA	HNDL		; LOAD HANDLE INTO AX
	LDX	HNDL+1
	JMP	(OPJMP)		; READ IN MEMORY BLOCK
.ENDIF
;*
;* CLEAR MEMORY BLOCK
;* ENTRY: AX = HANDLE
;* EXIT:  AX = HANDLE
;*
HMEM_CLR:	JSR	HMEM_PTR
	STA	DSTADDR
	STX	DSTADDR+1
	AND	#$FB		; POINT BACK TO HEADER
	STA	MPTR
	STX	MPTR+1
	LDY	#$01
	LDA	(MPTR),Y
	DEY
	TAX
	LDA	(MPTR),Y
	SEC			; SUBTRACT HEADER SIZE
.IFDEF	DEBUG_MEMMGR
	SBC	#$06
.ELSE
	SBC	#$04
.ENDIF
	BCS	:+
	DEX
:	JSR	MEMCLR		; CLEAR BLOCK
	LDA	HNDL		; RELOAD HANDLE INTO AX
	LDX	HNDL+1
	CLC
	RTS
;*
;* INCREMENT REFERENCE COUNT
;* ENTRY: AX = HANDLE
;* EXIT:  AX = HANDLE
;*
HMEM_REF_INC:	STA	HNDL
	STX	HNDL+1
.IFDEF	DEBUG
	JSR	HMEM_VALIDATE
	LDX	HNDL+1
.ENDIF
	LDY	#$00
	LDA	(HNDL),Y
	INY
	AND	#$F8
	STA	MPTR
	LDA	(HNDL),Y
	INY
	STA	MPTR+1
	LDA	(MPTR),Y
	CLC
	ADC	#$01
	STA	(MPTR),Y
	BCC	:+
	INY
	LDA	(MPTR),Y
	ADC	#$00
	STA	(MPTR),Y
:	LDA	HNDL
	RTS
;*
;* DECREMENT REFERENCE COUNT
;* ENTRY: AX = HANDLE
;* EXIT:  AX = REF COUNT
;*
HMEM_REF_DEC:	STA	HNDL
	STX	HNDL+1
.IFDEF	DEBUG
	JSR	HMEM_VALIDATE
.ENDIF
	LDY	#$00		; DEREF HANDLE
	LDA	(HNDL),Y
	INY
	AND	#$F8
	STA	MPTR
	LDA	(HNDL),Y
	INY
	STA	MPTR+1
	LDA	(MPTR),Y
	SEC			; DEC REF COUNT
	SBC	#$01
	STA	(MPTR),Y
	INY
	PHA
	LDA	(MPTR),Y
	TAX
	AND	#$7F
	SBC	#$00
	PHA
	ASL
	CPX	#$80		; RESTORE ACCESSED FLAG
	LSR
	STA	(MPTR),Y
	PLA
	TAX
	PLA
	RTS
;*
;* FREE MEMORY BLOCK
;*
HMEM_FREE:	STA	HNDL
	STX	HNDL+1
;.IFDEF	DEBUG_MEMMGR
.IF	 0
	JSR	PUTS
	.ASCIIZ	"FREE HANDLE: "
	LDA	HNDL+1
	LDX	HNDL
	JSR	PRNTAX
	LDA	#'-'
	JSR	COUT
	LDA	#'>'
	JSR	COUT
	LDY	#$00
	LDA	(HNDL),Y
	INY
	AND	#$F8
	TAX
	LDA	(HNDL),Y
	STA	$71
	STX	$70
	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	": LEN:"
	LDY	#$00
	LDA	($70),Y
	INY
	TAX
	LDA	($70),Y
	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	" REF:"
	LDY	#$02
	LDA	($70),Y
	INY
	TAX
	LDA	($70),Y
	JSR	PRNTAX
	JSR	CROUT
	LDA	HNDL
	LDX	HNDL+1
	JSR	HMEM_VALIDATE
	JSR	HFREE_VALIDATE
.ENDIF
	LDY	#$00		; FREE ALLOCED MEMORY BLOCK
	LDA	(HNDL),Y
	AND	#$F8
	STA	(HNDL),Y		; CLEAR ALLOCED FLAG
	INY
	STA	MPTR
	STY	GCNEEDED		; SET GC FLAG
	LDA	(HNDL),Y
	INY
	STA	MPTR+1
	LDA	HFREE		; ADD TO FREE LIST
	STA	(MPTR),Y
	INY
	LDA	HFREE+1
	STA	(MPTR),Y
	LDA	HNDL
	LDX	HNDL+1
	STA	HFREE
	STX	HFREE+1
.IFDEF	DEBUG_MEMMGR
	STA	HFREEBAK		; SAVE BACKUP
	STX	HFREEBAK+1
	JSR	HFREE_VALIDATE
.ENDIF
	RTS
;*
;* GET NEXT AVAILABLE HANDLE
;*
HMEM_NEW:	LDA	HUNUSED
	LDX	HUNUSED+1
	PHA
	LDY	#$00
	LDA	(HUNUSED),Y
	PHA
	INY
	LDA	(HUNUSED),Y
	BEQ	HNEW_ERR
	STA	HUNUSED+1
	PLA
	STA	HUNUSED
	PLA
	CLC
	RTS
HNEW_ERR:
	PERR	"OUT OF MEMORY HANDLES"
	JMP	THROW_INTERNALERR
;*
;* UNLINK FREE BLOCK FROM LIST
;*
HFREE_UNLINK:	CMP	HFREE
	BNE	:+
	CPX	HFREE+1
	BNE	:+
	LDY	#$00
	LDA	(HFREE),Y
	INY
	TAX
	LDA	(HFREE),Y
	INY
	STX	HFREE
	STA	HFREE+1
.IFDEF	DEBUG_MEMMGR
	STX	HFREEBAK		; SAVE BACKUP
	STA	HFREEBAK+1
.ENDIF	
	LDA	(HFREE),Y		; UNLINK FROM HEAD-OF-LIST
	INY
	TAX
	LDA	(HFREE),Y
	STX	HFREE
	STA	HFREE+1
.IFDEF	DEBUG_MEMMGR
	STX	HFREEBAK		; SAVE BACKUP
	STA	HFREEBAK+1
.ENDIF	
	RTS
:	STA	TMP
	STX	TMP+1
	LDA	HNDL		; SAVE HANDLE
	PHA
	LDA	HNDL+1
	PHA
	LDA	HFREE
	LDX	HFREE+1
LOOP_UNLINK:	STA	HNDL		; STORE HANDLE
	STX	HNDL+1
	LDY	#$01		; DEREF HANDLE FOR POINTER
	LDA	(HNDL),Y
	DEY
	TAX
	LDA	(HNDL),Y
	STA	HNDL
	STX	HNDL+1
	LDY	#$03		; CHECK NEXT HANDLE
	LDA	(HNDL),Y
.IFDEF	DEBUG
	BNE	:+
	PERR	"UNLINK FREE BLOCK NOT IN LIST"
	JMP	THROW_INTERNALERR
:
.ENDIF
	DEY
	TAX
	LDA	(HNDL),Y
	CMP	TMP
	BNE	LOOP_UNLINK
	CPX	TMP+1
	BNE	LOOP_UNLINK
	DEY			; HANDLES EQUAL
	LDA	(TMP),Y		; DEREF TMP
	DEY
	TAX
	LDA	(TMP),Y
	STA	TMPTR
	STX	TMPTR+1
	LDY	#$03
	LDA	(TMPTR),Y		; MOVE NEXT LINK TO PREVIOUS
	STA	(HNDL),Y
	DEY
	LDA	(TMPTR),Y
	STA	(HNDL),Y
.IFDEF	DEBUG
	JSR	HFREE_VALIDATE
.ENDIF
	PLA			; RESTORE HNDL
	STA	HNDL+1
	PLA
	STA	HNDL
	RTS
;*
;* ALLOCATE A MEMORY BLOCK
;* ENTRY: AX (LH) = SIZE
;*         Y      = INITIAL REF COUNT
;* EXIT:  AX      = HANDLE :: C == 0 
;*        ERR              :: C == 1
;*
HMEM_ALLOC:	STY	REFCNT
	CLC			; ADD SIZE OF HEADER (4 + 7)
.IFDEF	DEBUG_MEMMGR
	ADC	#$0D		; AND ROUND TO MULTIPLE OF 8 BYTES
.ELSE
	ADC	#$0B		; AND ROUND TO MULTIPLE OF 8 BYTES
.ENDIF
	BCC	:+
	INX
:	AND	#$F8
	STA	MLEN
	STX	MLEN+1
SEARCH_FREE:	LDA	HFREE		; SEARCH FREE LIST FOR AVAILABLE SPACE
	STA	HNDL
	LDA	HFREE+1
	STA	HNDL+1
	BEQ	ALLOC_ERR
	LDA	#$00		; CLEAR OUT BEST MATCH
	STA	HBEST
	STA	HBEST+1
LOOP_FREE:	LDY	#$01		; DEREF HANDLE
	LDA	(HNDL),Y
	DEY
	STA	MPTR+1
	LDA	(HNDL),Y	
	STA	MPTR
	LDA	(MPTR),Y
	INY
	SEC
	SBC	MLEN
	TAX
	LDA	(MPTR),Y
	SBC	MLEN+1
	BCC	NEXT_FREE
	BNE	:+
	CPX	#$00
	BEQ	MATCH_FREE
:	LDA	HBEST+1		; CHECK FOR EXISTING BEST MATCH
	BEQ	SAVE_BEST		; SAVE ONE WITH HIGHEST ADDRESS
	LDA	(HNDL),Y		; COMPARE FREE BLOCK ADDRESSES
	CMP	(HBEST),Y
	BCC	NEXT_FREE
	BNE	SAVE_BEST
	DEY
	LDA	(HNDL),Y
	CMP	(HBEST),Y
	BCC	NEXT_FREE
SAVE_BEST:	LDA	HNDL
	STA	HBEST
	LDA	HNDL+1
	STA	HBEST+1
NEXT_FREE:	LDY	#$02		; FOLLOW LINKED FREE LIST
	LDA	(MPTR),Y
	INY
	STA	HNDL
	LDA	(MPTR),Y
	STA	HNDL+1
	BNE	LOOP_FREE		; CONTINUE UNTIL NULL NEXT
	LDA	HBEST+1
	BNE	BEST_FREE
:	JSR	HMEM_COALESCE		; ATTEMPT EASY FREE SPACE COMBINE
	BCC	SEARCH_FREE
	JSR	HMEM_COMPACT		; ATTEMPT FREE SPACE COMPACTION
	BCC	:-
.IFDEF	SWAPPING
	JSR	HMEM_SWAPOUT		; ATTEMPT HARD MEMORY SWAPING
	BCC	:-
.ENDIF
ALLOC_ERR:
	LDA	#4		; OUT OF MEMORY
	JMP	SYSTHROW
MATCH_FREE:	LDA	HNDL		; EXACT FIT - USE IT
	LDX	HNDL+1
	JSR	HFREE_UNLINK
	LDY	#$00
	LDA	(HNDL),Y
	ORA	#$01		; SET ALLOCED FLAG
	STA	(HNDL),Y
	LDY	#$02		; SET INITIAL REFERENCE COUNT
	LDA	REFCNT
	STA	(MPTR),Y
	INY
	LDA	#$00
	STA	(MPTR),Y
.IFDEF	DEBUG
.IFDEF	DEBUG_MEMMGR
	LDA	MLEN		; WRITE FENCE
	LDX	MLEN+1
	SEC
	SBC	#$02
	BCS	:+
	DEX
:	CLC
	ADC	MPTR
	STA	MPTR
	TXA
	ADC	MPTR+1
	STA	MPTR+1
	LDY	#$00
	LDA	#$CA
	STA	(MPTR),Y
	INY
	LDA	#$FE
	STA	(MPTR),Y
.ENDIF
	JSR	HFREE_VALIDATE
	LDA	HNDL		; RETURN HANDLE, NO ERR
	LDX	HNDL+1
	JSR	HMEM_VALIDATE
.ENDIF
	LDA	HNDL		; RETURN HANDLE, NO ERR
	LDX	HNDL+1
	CLC
	RTS
BEST_FREE:	LDY	#$00		; DEREF FREE HANDLE
	LDA	(HBEST),Y
	INY
	STA	TMPTR
	LDA	(HBEST),Y
	DEY
	STA	TMPTR+1
	LDA	(TMPTR),Y
	SEC			; SUB MLEN FROM FREE SIZE
	SBC	MLEN
	STA	(TMPTR),Y
	INY
	LDA	(TMPTR),Y
	SBC	MLEN+1
	STA	(TMPTR),Y
	DEY			; CALC ALLOCED POINTER
	CLC
	LDA	TMPTR
	ADC	(TMPTR),Y
	INY
	STA	MPTR
	LDA	TMPTR+1
	ADC	(TMPTR),Y
	DEY			; SET LENGTH IN ALLOCED BLOCK
	STA	MPTR+1
	LDA	MLEN
	STA	(MPTR),Y
	INY
	LDA	MLEN+1
	STA	(MPTR),Y
	INY			; SET REFERENCE COUNT
	LDA	REFCNT
	STA	(MPTR),Y
	INY
	LDA	#$00
	STA	(MPTR),Y
	JSR	HMEM_NEW		; GET NEW HANDLE
.IFDEF	DEBUG
	BCC	:+
	JMP	ALLOC_ERR
:
.ELSE
	BCS	ALLOC_ERR
.ENDIF
	STA	HNDL
	STX	HNDL+1
	LDY	#$00		; UPDATE HANDLE POINTERS
	LDA	MPTR
	ORA	#$01		; SET ALLOCED FLAG
	STA	(HNDL),Y		; SET POINTER IN HANDLE TABLE
	INY
	LDA	MPTR+1
	STA	(HNDL),Y
.IFDEF	DEBUG
.IFDEF	DEBUG_MEMMGR
	LDA	MLEN		; WRITE FENCE
	LDX	MLEN+1
	SEC
	SBC	#$02
	BCS	:+
	DEX
:	CLC
	ADC	MPTR
	STA	MPTR
	TXA
	ADC	MPTR+1
	STA	MPTR+1
	LDY	#$00
	LDA	#$CA
	STA	(MPTR),Y
	INY
	LDA	#$FE
	STA	(MPTR),Y
.ENDIF
	LDA	HNDL
	LDX	HNDL+1
	JSR	HMEM_VALIDATE
	LDX	HNDL+1
.ENDIF
	LDA	HNDL		; RETURN HANDLE, NO ERR
	CLC
	RTS
;*
;* ALLOC CODE MEMORY
;* ENTRY: AX (LH) = SIZE
;*        Y       = INITIAL REF COUNT
;* EXIT:  AX      = HANDLE :: C == 0 
;*        ERR              :: C == 1
;*
HMEM_ALLOC_CODE: JSR	HMEM_ALLOC
	BCS	:+
	LDY	#$00
	LDA	(HNDL),Y
	ORA	#$04		; SET CODE FLAG IN ALLOC BLOCK FLAGS
	STA	(HNDL),Y
	LDA	HNDL
:	RTS
	
;*
;* ALLOCATE A MEMORY BLOCK AT FIXED ADDRESS
;* ENTRY: X   = PAGE ADDRESS
;*        A   = SIZE IN PAGES
;*        Y   = INITIAL REF COUNT
;* EXIT:  AX  = HANDLE :: C == 0 
;*        ERR          :: C == 1
;*
HMEM_ALLOC_FIXED: STY	REFCNT
	STA	MEND+1
.IFDEF	DEBUG_MEMMGR
	LDY	#$02
	STY	MEND
	LDY	#$00
.ELSE
	LDY	#$00
	STY	MEND
.ENDIF
	TXA
	CLC
	ADC	MEND+1
	STA	MEND+1
	LDA	MEND
	SEC
	SBC	#$04		; SUBTRACT HEADER SIZE
	AND	#$F8		; MAKE ADDRESS 8 BYTE BOUNDARY
	STA	MPTR		; SAVE IN MPTR
	TXA
	SBC	#$00	
	STA	MPTR+1
	LDA	MEND		; ROUND ENDING ADDRESS UP
	CLC			; TO 8 BYTE BOUNDARY
	ADC	#$07
	AND	#$F8
	STA	MEND
	TYA			; Y = 0
	ADC	MEND+1
	STA	MEND+1
	LDA	MEND
	SEC			; CALC LEN = MEND - MPTR
	SBC	MPTR
	STA	MLEN
	LDA	MEND+1
	SBC	MPTR+1
	STA	MLEN+1		; SAVE LENGHT IN BYTES
SEARCH_FIXED:	LDA	HFREE		; SEARCH FREE LIST FOR AVAILABLE SPACE
	STA	HNDL
	LDA	HFREE+1
	STA	HNDL+1
	BEQ	ALLOC_FIXED_ERR
LOOP_FIXED:	LDY	#$01		; DEREF HANDLE
	LDA	(HNDL),Y
	DEY
	STA	TMPTR+1
	LDA	(HNDL),Y
	STA	TMPTR
	LDA	MPTR		; COMPARE STARTING ADDRESSES
	CMP	TMPTR
	LDA	MPTR+1
	SBC	TMPTR+1
	BCC	NEXT_FIXED
	LDA	TMPTR		; COMPARE ENDING ADDRESSES
	CLC			; CALC END ADDRESS OF FREE BLOCK
	ADC	(TMPTR),Y
	INY
	TAX
	LDA	TMPTR+1
	ADC	(TMPTR),Y
	TAY
	TXA
	SEC
	SBC	MEND
	TAX
	TYA
	SBC	MEND+1
	BCS	FOUND_FIXED
NEXT_FIXED:	LDY	#$02		; FOLLOW LINKED FREE LIST
	LDA	(TMPTR),Y
	INY
	STA	HNDL
	LDA	(TMPTR),Y
	STA	HNDL+1
	BNE	LOOP_FIXED		; CONTINUE UNTIL NULL NEXT
;:
;	JSR	HMEM_COALESCE		; ATTEMPT EASY FREE SPACE COMBINE
;	BCC	SEARCH_FIXED
;	JSR	HMEM_COMPACT		; ATTEMPT FREE SPACE COMPACTION
;	BCC	:-
;.IFDEF	SWAPPING
;	JSR	HMEM_SWAPOUT		; ATTEMPT HARD MEMORY SWAPING
;	BCC	:-
;.ENDIF
ALLOC_FIXED_ERR:
.IFDEF	DEBUG_MEMMGR
	PERR	"ALLOC_FIXED PAGES UNAVAILABLE"
;	JSR	HMEM_DUMP
;	JSR	KBWAIT
.ENDIF
	SEC			; RETURN WITH ERR
	RTS
FOUND_FIXED:	BNE	END_FREE_FIXED		; CHECK FOR EQUAL END ADDRESS
	CPX	#$00
	BEQ	FRONT_FREE_FIXED
END_FREE_FIXED:	LDY	#$01		; SAVE SIZE IN NEW FREE BLOCK
	STA	(MEND),Y
	DEY
	TXA
	STA	(MEND),Y
	LDY	#$02
	LDA	HFREE		; INSERT NEW FREE BLOCK INTO LIST
	STA	(MEND),Y
	INY
	LDA	HFREE+1
	STA	(MEND),Y
	JSR	HMEM_NEW		; GET NEW HANDLE FOR END FREE BLOCK
	BCS	ALLOC_FIXED_ERR
	STA	HFREE
	STX	HFREE+1
.IFDEF	DEBUG_MEMMGR
	STA	HFREEBAK		; SAVE BACKUP
	STX	HFREEBAK+1
.ENDIF	
	LDY	#$00		; SAVE ADDRESS IN POINTER TABLE
	LDA	MEND
	STA	(HFREE),Y
	INY
	LDA	MEND+1
	STA	(HFREE),Y
FRONT_FREE_FIXED: LDA	MPTR+1
	CMP	TMPTR+1
	BNE	:+
	LDA	MPTR
	CMP	TMPTR
	BEQ	FRONT_EQUAL_FIXED
:	LDY	#$00		; SET SIZE OF FRONT FREE BLOCK
	LDA	MPTR
	SEC
	SBC	TMPTR
	STA	(TMPTR),Y
	INY
	LDA	MPTR+1
	SBC	TMPTR+1
	STA	(TMPTR),Y
	JSR	HMEM_NEW		; GET NEW HANDLE FOR ALLOCED BLOCK
	BCS	ALLOC_FIXED_ERR
	STA	HNDL
	STX	HNDL+1
	LDY	#$00		; UPDATE POINTER IN HANDLE TABLE
	LDA	MPTR
	ORA	#$03		; SET FIXED AND ALLOCED FLAGS
	STA	(HNDL),Y
	INY
	LDA	MPTR+1
	STA	(HNDL),Y
	DEY
	BEQ	SET_ALLOC_FIXED		; FILL IN THE REST
FRONT_EQUAL_FIXED: LDA	HNDL		; SAME START ADDRESS FOR FREE AND ALLOC BLOCK
	LDX	HNDL+1		; RE-USE FREE BLOCK
	JSR	HFREE_UNLINK		; UNLINK FREE BLOCK
	LDY	#$00		; SET FIXED AND ALLOCED FLAGS
	LDA	(HNDL),Y
	ORA	#$03
	STA	(HNDL),Y
SET_ALLOC_FIXED: LDA	MLEN		; SET BLOCK LENGTH	
	STA	(MPTR),Y
	INY
	LDA	MLEN+1
	STA	(MPTR),Y
	LDY	#$02		; SET REFERENCE COUNT
	LDA	REFCNT
	STA	(MPTR),Y
	INY
	LDA	#$00
	STA	(MPTR),Y
.IFDEF	DEBUG
.IFDEF	DEBUG_MEMMGR
	LDA	MLEN		; WRITE FENCE
	LDX	MLEN+1
	SEC
	SBC	#$02
	BCS	:+
	DEX
:	CLC
	ADC	MPTR
	STA	MPTR
	TXA
	ADC	MPTR+1
	STA	MPTR+1
	LDY	#$00
	LDA	#$CA
	STA	(MPTR),Y
	INY
	LDA	#$FE
	STA	(MPTR),Y
.ENDIF
	LDA	HNDL		; RETURN HANDLE, NO ERR
	LDX	HNDL+1
	JSR	HMEM_VALIDATE
.ENDIF
	LDA	HNDL		; RETURN HANDLE, NO ERR
	LDX	HNDL+1
	CLC
	RTS
;*
;* LOCK MEMORY BLOCK IN PLACE
;* ENTRY: AX = HANDLE
;* EXIT:  AX = POINTER
;*
HMEM_LOCK:	JSR	HMEM_PTR		; MAKE SURE MEM BLOCK PRESENT
	LDY	#$00
	AND	#$F8
	ORA	#$03		; CHANGE FLAGS TO LOCKED
	STA	(HNDL),Y		; AND SAVE
	AND	#$F8
	ORA	#$04
	RTS
;*
;* UNLOCK MEMORY BLOCK
;* ENTRY: AX = HANDLE
;*
HMEM_UNLOCK:	STA	HNDL
	STX	HNDL+1
.IFDEF	DEBUG
	JSR	HMEM_VALIDATE
.ENDIF
	LDY	#$00
	LDA	(HNDL),Y
.IFDEF	DEBUG
	TAX
	AND	#$07
	CMP	#$03
	BEQ	:+
	PERR	"UNLOCKING UNLOCKED HANDLE"
	JMP	THROW_INTERNALERR
:	TXA
.ENDIF
	AND	#$F8
	ORA	#$01
	STA	(HNDL),Y
	RTS
;*
;* UNLOCK CODE MEMORY BLOCK
;* ENTRY: AX = HANDLE
;*
HMEM_UNLOCK_CODE: JSR	HMEM_UNLOCK
	ORA	#$04
	STA	(HNDL),Y
	RTS
;*******************************
;*
;* GARBAGE COLLECTION ROUTINES
;*
;*******************************
;*
;* GARBAGE COLLECT
;* ENTRY: AX = MAX ITERATIONS
;* EXIT:  CFLAG=1, NOTHING COLLECTED; CFLAG=0, GARBAGE COLLECTED
;*
HMEM_GC:	ADC	#$01
	INX
	STA	GCCNT
	STX	GCCNT+1
.IFDEF	DEBUG_MEMMGR
	JSR	PUTSLN
	.ASCIIZ	"STARTING GARBAGE COLLECTION:"
	JSR	HMEM_DUMP
	JSR	KBWAIT
.ENDIF
LOOPGC:	DEC	GCCNT
	BNE	:+
	DEC	GCCNT+1
	BNE	:+
	RTS
:	JSR	HMEM_COALESCE		; ATTEMPT EASY FREE SPACE COMBINE
	BCC	LOOPGC
	JSR	HMEM_COMPACT		; ATTEMPT FREE SPACE COMPACTION
	BCC	LOOPGC
.IFDEF	DEBUG_MEMMGR
	JSR	PUTSLN
	.ASCIIZ	"FINISHED GARBAGE COLLECTION:"
	JSR	HMEM_DUMP
	JSR	KBWAIT
.ENDIF
	RTS
GCCNT:	.WORD	$0000
;*
;* COALESCE ADJACENT FREE BLOCKS
;* ENTRY:
;* EXIT:  CFLAG=1, NOTHING COALESCED; CFLAG=0, FREE MEM COALESCED
;*
HMEM_COALESCE:	
.IFDEF	DEBUG_MEMMGR
	PERR	"HMEM_COALESCING..."
.ENDIF
	LDA	HFREE		; SEARCH FREE LIST FOR BLOCKS TO COMBINE
	STA	GCHNDL
	LDA	HFREE+1
	STA	GCHNDL+1
	BEQ	NOTCOALESCED
SEARCHCOALESCE: LDY	#$01		; DEREF HANDLE
	LDA	(GCHNDL),Y
	DEY
	STA	GCMPTR+1
	LDA	(GCHNDL),Y
	STA	GCMPTR
	CLC
	ADC	(GCMPTR),Y
	INY
	STA	GCMEND
	LDA	GCMPTR+1
	ADC	(GCMPTR),Y
	STA	GCMEND+1
	LDA	HFREE
	STA	GCTMP
	LDA	HFREE+1
	STA	GCTMP+1
LOOPCOALESCE:	LDY	#$00		; COMPARE THIS FREE BLOCK ADDRESS
	LDA	(GCTMP),Y		; WITH END OF FREE BLOCK BEING CHECKED
	INY
	TAX
	LDA	(GCTMP),Y
	CMP	GCMEND+1
	BNE	NEXTCOAL
	CPX	GCMEND
	BEQ	MATCHCOALESCE		; A MATCH, COMBINE BLOCKS
NEXTCOAL:	INY
	STA	GCTMP+1		; GET POINTER TO FREE BLOCK
	STX	GCTMP
	LDA	(GCTMP),Y		; GET NEXT FREE BLOCK HANDLE
	INY
	TAX
	LDA	(GCTMP),Y
	STX	GCTMP
	STA	GCTMP+1
	BNE	LOOPCOALESCE		; KEEP CHECKING
	LDA	(GCMPTR),Y		; NEXT FREE BLOCK TO COMPARE WITH
	BEQ	NOTCOALESCED
	STA	GCHNDL+1
	DEY
	LDA	(GCMPTR),Y
	STA	GCHNDL
	JMP	SEARCHCOALESCE
NOTCOALESCED:	SEC
	RTS
MATCHCOALESCE:
.IFDEF	DEBUG_MEMMGR
	JSR	PUTS
	.ASCIIZ	"COMBINING "
	LDA	GCHNDL+1
	LDX	GCHNDL
	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	" AND "
	LDA	GCTMP+1
	LDX	GCTMP
	JSR	PRNTAX
	JSR	CROUT
.ENDIF
	LDA	GCTMP		; UNLINK FREE BLOCK FROM LIST
	LDX	GCTMP+1
	JSR	HFREE_UNLINK
	LDY	#$00		; RETURN UNREFERENCED HANDLE BACK TO UNUSED LIST
	LDA	(GCTMP),Y
	PHA			; SAVE MEMORY POINTER FOR LATER
	LDA	HUNUSED
	STA	(GCTMP),Y
	INY
	LDA	(GCTMP),Y
	PHA
	LDA	HUNUSED+1
	STA	(GCTMP),Y
	DEY
	LDA	GCTMP
	STA	HUNUSED
	LDA	GCTMP+1
	STA	HUNUSED+1
	PLA			; RECOVER MEMORY POINTER
	STA	GCTMP+1
	PLA
	STA	GCTMP
	LDA	(GCMPTR),Y		; ADD SIZE OF NEXT FREE BLOCK TO CURRENT
	CLC	
	ADC	(GCTMP),Y
	STA	(GCMPTR),Y
	INY
	LDA	(GCMPTR),Y
	ADC	(GCTMP),Y
	STA	(GCMPTR),Y
.IFDEF	DEBUG_MEMMGR
	JSR	HFREE_VALIDATE
.ENDIF
	CLC
	RTS
;*
;* COMPACT MEMORY BY MOVING ALLOCED BLOCKS TO TOP OF HEAP
;* ENTRY:
;* EXIT:  CFLAG=1, NOTHING COMPACTED; CFLAG=0, MEM COMPACTED
;*
HMEM_COMPACT:	
.IFDEF	DEBUG_MEMMGR
	PERR	"HMEM_COMPACTING..."
.ENDIF
	LDY	#$00
	STY	GCBEST+1
	STY	GCNEIGHBOR+1
	LDA	HFREE		; SEARCH FREE LIST FOR BLOCKS TO MOVE
	STA	GCHNDL
	LDA	HFREE+1
	STA	GCHNDL+1
	BNE	SEARCHCOMPACT
	SEC
	RTS
SEARCHCOMPACT:	LDY	#$01		; DEREF HANDLE
	LDA	(GCHNDL),Y
	DEY
	STA	GCMPTR+1
	LDA	(GCHNDL),Y
	STA	GCMPTR		; CALC START AND END ADDRESSES FOR
	LDA	#<HTBL		; START SEARCH FOR ALLOCATED BLOCKS
	STA	GCTMP		; AT BEGINNING OF TABLE
	LDA	#>HTBL
	STA	GCTMP+1
LOOPCOMPACT:	LDY	#$00
	LDA	(GCTMP),Y
	TAX
	LSR			; CHECK ALLOCED FLAG
	BCC	NEXTCOMP		; SKIP FREE BLOCK
	AND	#$03
	CMP	#$01		; CHECK IF LOCKED
	BEQ	NEXTCOMP
	TXA
	AND	#$F8		; MASK OFF FLAGS
	TAX
	INY
	LDA	(GCTMP),Y		; COMPARE FOR LOWER ADDRESS THAN FREE BLOCK
	CMP	GCMPTR+1
	BNE	:+	
	CPX	GCMPTR
:	BCS	NEXTCOMP
	STA	GCMEND+1		; CALC END ADDRESS OF BLOCK
	STX	GCMEND		; IF IT MATCHES START OF FREE BLOCK
	DEY			; THEN IT CAN BE SHIFTED UP
	TXA
	CLC
	ADC	(GCMEND),Y
	INY
	TAX
	LDA	GCMEND+1
	ADC	(GCMEND),Y
	CMP	GCMPTR+1
	BNE	:+
	CPX	GCMPTR
	BNE	:+
	LDA	GCTMP		; SAVE NEAREST NEIGHBOR
	STA	GCNEIGHBOR
	LDA	GCTMP+1
	STA	GCNEIGHBOR+1
;	BEQ	SHIFTNEIGHBOR		; MOVE NEXT-DOOR NEIGHBOR UP
:	LDA	(GCMPTR),Y		; ELSE COMPARE SIZE OF BLOCK TO FREE BLOCK
	CMP	(GCMEND),Y		; IT MUST BE EQUAL TO SIZE
	BNE	NEXTCOMP		; OF FREE BLOCK
	DEY
	LDA	(GCMPTR),Y
	CMP	(GCMEND),Y
	BNE	NEXTCOMP
CHECKBESTCOMP:	LDA	GCBEST+1		; ONLY REPLACE BEST MATCH IF AT A HIGER ADDRESS
	BEQ	:+		; ASSUMING HIGHER ADDRESSES ARE OLDER AND
	INY			; WILL BE FREED LAST
	LDA	GCMEND+1
	CMP	(GCBEST),Y
	BCC	NEXTCOMP		; CURRENT BEST IS HIGHER
	BNE	:+
	DEY
	LDA	GCMEND
	CMP	(GCBEST),Y
	BCC	NEXTCOMP		; CURRENT BEST IS HIGER
:	LDA	GCTMP
	STA	GCBEST
	LDA	GCTMP+1
	STA	GCBEST+1
NEXTCOMP:	LDA	GCTMP		; MOVE TO NEXT HANDLE IN TABLE
	CLC	
	ADC	#$02
	STA	GCTMP
	BNE	LOOPCOMPACT
	INC	GCTMP+1
	LDX	GCTMP+1
	CPX	#>HTBL_END
	BNE	LOOPCOMPACT
	LDA	GCBEST+1		; LOOK FOR SAVED BEST MATCH
	BNE	SWAPBEST
	LDA	GCNEIGHBOR+1		; LOOK FOR NEAREST NEIGHBOR
	BNE	SHIFTNEIGHBOR
	LDY	#$03
	LDA	(GCMPTR),Y		; NEXT FREE BLOCK TO COMPARE WITH
	BEQ	EXITCOMPACT
	STA	GCHNDL+1
	DEY
	LDA	(GCMPTR),Y
	STA	GCHNDL
	JMP	SEARCHCOMPACT
EXITCOMPACT:	SEC
	RTS
SHIFTNEIGHBOR:	STA	GCBEST+1
	LDA	GCNEIGHBOR
	STA	GCBEST
.IFDEF	DEBUG_MEMMGR
	JSR	PUTS
	.ASCIIZ	"SHIFT NEIGHBOR "
	LDA	GCNEIGHBOR+1
	LDX	GCNEIGHBOR
	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	" OVER "
	LDA	GCHNDL+1
	LDX	GCHNDL
	JSR	PRNTAX
	JSR	CROUT
	LDA	GCNEIGHBOR
	LDX	GCNEIGHBOR+1
	JSR	HMEM_VALIDATE
	LDY	#$01
.ENDIF
	LDY	#$00		; SET COPY POINTERS
	LDA	(GCBEST),Y
	AND	#$F8
	STA	SRCADDR
	CLC			; ADD SIZE OF FREE BLOCK
	ADC	(GCMPTR),Y		; FOR DESTINATION
	STA	DSTADDR
	INY
	LDA	(GCBEST),Y
	STA	SRCADDR+1
	ADC	(GCMPTR),Y
	STA	DSTADDR+1
	BNE	MOVEBLOCKS
SWAPBEST:
.IFDEF	DEBUG_MEMMGR
	JSR	PUTS
	.ASCIIZ	"SWAPPING BEST "
	LDA	GCBEST+1
	LDX	GCBEST
	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	" WITH "
	LDA	GCHNDL+1
	LDX	GCHNDL
	JSR	PRNTAX
	JSR	CROUT
	LDA	GCBEST
	LDX	GCBEST+1
	JSR	HMEM_VALIDATE	
.ENDIF
	LDY	#$00		; SET COPY POINTERS
	LDA	(GCBEST),Y
	AND	#$F8
	STA	SRCADDR
	LDA	GCMPTR
	STA	DSTADDR
	INY
	LDA	(GCBEST),Y
	STA	SRCADDR+1
	LDA	GCMPTR+1
	STA	DSTADDR+1
MOVEBLOCKS:
	LDY	#$00		; SAVE FREE BLOCK INFO
:	LDA	(GCMPTR),Y
	PHA
	INY
	CPY	#$04
	BNE	:-	
	LDY	#$00		; UPDATE ALLOC BLOCK POINTER
	LDA	(GCBEST),Y
	AND	#$07		; COPY ALLOC FLAGS
	ORA	DSTADDR
	STA	(GCBEST),Y
	LDA	SRCADDR
	STA	(GCHNDL),Y		; UPDATE FREE BLOCK POINTER
	STA	GCMPTR
	INY
	LDA	DSTADDR+1
	STA	(GCBEST),Y
	LDA	SRCADDR+1
	STA	(GCHNDL),Y
	STA	GCMPTR+1
	LDA	(SRCADDR),Y		; SIZE OF ALLOCED BLOCK TO MOVE
	DEY
	TAX
	LDA	(SRCADDR),Y
	JSR	MEMCPY		; MOVE ALLOCED BLOCK UP	
	LDY	#$03		; RESTORE FREE BLOCK INFO
:	PLA
	STA	(GCMPTR),Y
	DEY
	BPL	:-
.IFDEF	DEBUG_MEMMGR
	LDA	GCBEST
	LDX	GCBEST+1
	JSR	HMEM_VALIDATE	
	JSR	HFREE_VALIDATE
.ENDIF
	CLC
	RTS
.IFDEF	SWAPPING
;*
;* SWAP OUT MEMORY BLOCK
;*
HMEM_SWAPOUT:	SEC
	RTS
.ENDIF
.IFDEF	DEBUG
;*
;* VALIDATE HANDLE/POINTER
;*
HMEM_VALIDATE:	STA	$70
	STX	$71
	CPX	#>HTBL
	BCC	BADHNDL
	CPX	#$BF
	BCS	BADHNDL
	AND	#$01
	BNE	BADHNDL
	LDY	#$01
	LDA	($70),Y
	DEY
	TAX
	LDA	($70),Y
	TAY
	AND	#$F8
	CPX	#>INIT_START
	BCC	BADHNDL
	BNE	:+
	CMP	#<INIT_START
	BCC	BADHNDL
:	CPX	#>HTBL
	BCC	:+
	BNE	BADHNDL
	CMP	#<HTBL
	BCS	BADHNDL
:	TYA
	AND	#$01
	BEQ	BADHNDL
.IFDEF	DEBUG_MEMMGR
	TYA
	AND	#$F8
	STA	$72
	STX	$73
	TXA
	LDY	#$01
	LDA	($72),Y
	DEY
	TAX
	LDA	($72),Y
	SEC
	SBC	#$02
	BCS	:+
	DEX
:	CLC
	ADC	$72
	STA	$72
	TXA
	ADC	$73
	STA	$73
	LDA	#$CA
	CMP	($72),Y
	BNE	BADFENCE
	INY
	LDA	#$FE
	CMP	($72),Y
	BNE	BADFENCE
	RTS
BADFENCE:	PERR	"CORRUPTED FENCE:"
	JMP	PRNTBADHNDL
.ENDIF
	RTS
BADHNDL:	PERR	"INVALID MEMORY HANDLE/POINTER:"
PRNTBADHNDL:	LDA	$71
	LDX	$70
	JSR	PRNTAX
	LDA	#'/'
	JSR	COUT
	LDY	#$00
	LDA	($70),Y
	INY
	TAX
	LDA	($70),Y
	JSR	PRNTAX
	JSR	CROUT
.IFDEF	DEBUG_MEMMGR
	JSR	HMEM_DUMP
.ENDIF
	LDA	BADHNDLBRK		; STOP RECURSIVE BAD HANDLES
	BEQ	BADHNDLBRK
	DEC	BADHNDLBRK
	JMP	THROW_INTERNALERR
BADHNDLBRK:	.BYTE	$01
HFREEBAK:	.WORD	$0000
HFREE_VALIDATE:	LDA	HFREE
.IFDEF	DEBUG_MEMMGR
	CMP	HFREEBAK
	BNE	BADFREE
.ENDIF
	STA	$70
	LDX	HFREE+1
.IFDEF	DEBUG_MEMMGR
	CPX	HFREEBAK+1
	BNE	BADFREE
.ENDIF
	STX	$71
VALFREE:	CPX	#$00
	BEQ	VALFREEDONE
	CPX	#>HTBL
	BCC	BADFREE
	CPX	#$BF
	BCS	BADFREE
	AND	#$01
	BNE	BADFREE
	LDY	#$01
	LDA	($70),Y
	DEY
	TAX
	LDA	($70),Y
	AND	#$F8
	CPX	#>INIT_START
	BCC	BADFREE
	BNE	:+
	CMP	#<INIT_START
	BCS	:+
	JMP	BADFREE
:	CPX	#>HTBL
	BCC	:+
	BEQ	:+
	JMP	BADFREE
	CMP	#<HTBL
	BCC	:+
	JMP	BADFREE
:	LDY	#$01		; DEREF HANDLE
	LDA	($70),Y
	DEY
	TAX
	LDA	($70),Y
	STA	$72
	STX	$73
	LDY	#$03
	LDA	($72),Y
	DEY
	TAX
	LDA	($72),Y
	STA	$70
	STX	$71
	JMP	VALFREE
VALFREEDONE:	RTS
BADFREE:	PERR	"CORRUPTED FREE LIST @:"
	LDA	$71
	LDX	$70
	JSR	PRNTAX
	JSR	CROUT
.IFDEF	DEBUG_MEMMGR
	JSR	HMEM_DUMP
.ENDIF
	JMP	THROW_INTERNALERR
.ENDIF
.IFDEF	DEBUG_MEMMGR
;*
;* PRINT OUT MEMORY MANAGER DETAILS
;*
	.EXPORT	HMEM_DUMP
HMEM_DUMP:	JSR	CROUT
	JSR	PUTSLN
	.ASCIIZ	"MEMORY MANAGER STATE"
	JSR	PUTSLN
	.ASCIIZ	"===================="
	JSR	PUTSLN
	.ASCIIZ	"FREE LIST"
	JSR	PUTSLN
	.ASCIIZ	"---------"
	LDA	HFREE
	STA	HNDL
	LDA	HFREE+1
	STA	HNDL+1
FREE_DUMP:	LDX	HNDL
	LDA	HNDL+1
	BNE	:+
	JMP	FREE_DONE
:	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	"->"
	LDY	#$00
	LDA	(HNDL),Y
	INY
	STA	MPTR
	TAX
	LDA	(HNDL),Y
	STA	MPTR+1
	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	": LEN:"
	LDY	#$00
	LDA	(MPTR),Y
	INY
	TAX
	LDA	(MPTR),Y
	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	" NEXT:"
	LDY	#$02
	LDA	(MPTR),Y
	INY
	TAX
	STA	HNDL
	LDA	(MPTR),Y
	STA	HNDL+1
	JSR	PRNTAX
	JSR	CROUT
	JMP	FREE_DUMP
FREE_DONE:	LDA	#<HTBL
	STA	HNDL
	LDA	#>HTBL
	STA	HNDL+1
	JSR	PUTSLN
	.ASCIIZ	"ALLOCATED BLOCKS"
	JSR	PUTSLN
	.ASCIIZ	"----------------"
ALLOC_DUMP:	LDY	#$00		; MAKE UNUSED HANDLE LIST
	LDA	(HNDL),Y
	AND	#$01
	BEQ	NEXT_ADUMP		; NOT ALLOCATED
	LDX	HNDL
	LDA	HNDL+1
	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	"->"
	LDY	#$00
	LDA	(HNDL),Y
	INY
	TAX
	AND	#$F8
	STA	MPTR
	LDA	(HNDL),Y
	STA	MPTR+1
	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	": LEN:"
	LDY	#$00
	LDA	(MPTR),Y
	INY
	TAX
	LDA	(MPTR),Y
	JSR	PRNTAX
	JSR	PUTS
	.ASCIIZ	" REF:"
	LDY	#$02
	LDA	(MPTR),Y
	INY
	TAX
	LDA	(MPTR),Y
	JSR	PRNTAX
	JSR	CROUT
NEXT_ADUMP:	LDA	HNDL
	CLC
	ADC	#$02
	STA	HNDL
	BNE	ALLOC_DUMP
	INC	HNDL+1
	LDA	HNDL+1
	CMP	#>HTBL_END
	BNE	ALLOC_DUMP
	JSR	CROUT
	RTS
.ENDIF